<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

class SetupController extends Controller
{
    /**
     * Show the setup page
     */
    public function index()
    {
        // Only allow setup if not already configured
        $setupComplete = File::exists(storage_path('app/setup_complete.flag'));
        
        return view('setup.index', compact('setupComplete'));
    }

    /**
     * Run the setup process via web interface
     */
    public function setup(Request $request)
    {
        $results = [];
        $hasErrors = false;

        // Step 1: Create storage link
        try {
            $publicStoragePath = public_path('storage');
            
            // Remove existing link if it exists and is broken
            if (File::exists($publicStoragePath)) {
                if (is_link($publicStoragePath) && !File::exists(readlink($publicStoragePath))) {
                    unlink($publicStoragePath);
                } elseif (is_dir($publicStoragePath) && !is_link($publicStoragePath)) {
                    File::deleteDirectory($publicStoragePath);
                }
            }
            
            // Try to create storage link
            if (!File::exists($publicStoragePath)) {
                try {
                    Artisan::call('storage:link');
                    $results['storage'] = ['status' => 'success', 'message' => 'Storage link created successfully'];
                } catch (\Exception $e) {
                    // Fallback: Copy files instead of symlink for shared hosting
                    $this->copyStorageFiles();
                    $results['storage'] = ['status' => 'success', 'message' => 'Storage files copied (symlink not available)'];
                }
            } else {
                $results['storage'] = ['status' => 'success', 'message' => 'Storage link already exists'];
            }
        } catch (\Exception $e) {
            $results['storage'] = ['status' => 'error', 'message' => 'Storage setup failed: ' . $e->getMessage()];
            $hasErrors = true;
        }

        // Step 2: Clear caches
        try {
            Artisan::call('config:clear');
            Artisan::call('cache:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            $results['cache'] = ['status' => 'success', 'message' => 'All caches cleared successfully'];
        } catch (\Exception $e) {
            $results['cache'] = ['status' => 'error', 'message' => 'Cache clearing failed: ' . $e->getMessage()];
            $hasErrors = true;
        }

        // Step 3: Check environment file
        try {
            if (!File::exists(base_path('.env'))) {
                if (File::exists(base_path('.env.example'))) {
                    File::copy(base_path('.env.example'), base_path('.env'));
                    $results['env'] = ['status' => 'warning', 'message' => 'Environment file created. Please update database settings.'];
                } else {
                    $results['env'] = ['status' => 'error', 'message' => 'No .env.example file found'];
                    $hasErrors = true;
                }
            } else {
                $results['env'] = ['status' => 'success', 'message' => 'Environment file exists'];
            }
        } catch (\Exception $e) {
            $results['env'] = ['status' => 'error', 'message' => 'Environment check failed: ' . $e->getMessage()];
            $hasErrors = true;
        }

        // Step 4: Create directories if they don't exist
        try {
            $directories = [
                storage_path('app/public'),
                storage_path('logs'),
                public_path('storage'),
            ];

            foreach ($directories as $dir) {
                if (!File::exists($dir)) {
                    File::makeDirectory($dir, 0755, true);
                }
            }
            $results['directories'] = ['status' => 'success', 'message' => 'Required directories created'];
        } catch (\Exception $e) {
            $results['directories'] = ['status' => 'error', 'message' => 'Directory creation failed: ' . $e->getMessage()];
            $hasErrors = true;
        }

        // Mark setup as complete if no critical errors
        if (!$hasErrors) {
            File::put(storage_path('app/setup_complete.flag'), 'Setup completed on ' . now());
        }

        return response()->json([
            'success' => !$hasErrors,
            'results' => $results,
            'message' => $hasErrors ? 'Setup completed with some issues' : 'Setup completed successfully!'
        ]);
    }

    /**
     * Fallback method to copy storage files instead of creating symlink
     * This works on shared hosting where symlinks are disabled
     */
    private function copyStorageFiles()
    {
        $sourceDir = storage_path('app/public');
        $targetDir = public_path('storage');

        if (!File::exists($sourceDir)) {
            File::makeDirectory($sourceDir, 0755, true);
        }

        if (!File::exists($targetDir)) {
            File::makeDirectory($targetDir, 0755, true);
        }

        // Copy all files from storage/app/public to public/storage
        if (File::exists($sourceDir)) {
            $files = File::allFiles($sourceDir);
            foreach ($files as $file) {
                $relativePath = str_replace($sourceDir, '', $file->getPathname());
                $targetFile = $targetDir . $relativePath;
                
                // Create directory if it doesn't exist
                $targetDirectory = dirname($targetFile);
                if (!File::exists($targetDirectory)) {
                    File::makeDirectory($targetDirectory, 0755, true);
                }
                
                File::copy($file->getPathname(), $targetFile);
            }
        }
    }

    /**
     * Reset setup (for testing)
     */
    public function reset()
    {
        if (File::exists(storage_path('app/setup_complete.flag'))) {
            File::delete(storage_path('app/setup_complete.flag'));
        }

        return redirect()->route('setup.index')->with('message', 'Setup reset. You can run setup again.');
    }
}
