<?php

namespace App\Livewire\Admin\ManageDeposits;

use App\Models\Deposit;
use App\Models\Settings;
use App\Models\Transaction;
use App\Notifications\User\DepositConfirmedNotification;
use App\Services\ReferralCommisionService;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.admin')]
class Records extends Component
{
    use WithPagination;
    use LivewireAlert;

    #[Url(except: '')]
    public $search = '';
    public $order = 'desc';
    public $status = 'All';
    public $perPage = 10;
    public $toDate = '';
    public $fromDate = '';
    public $deleteId = '';

    #[On('close-deposit-modal')]
    public function render()
    {
        return view('livewire.admin.manage-deposits.records', [
            'deposits' => Deposit::search($this->search)
                ->status($this->status)
                ->orderBy('id', $this->order)
                ->with('user:id,name,username')
                ->date($this->fromDate, $this->toDate)
                ->paginate($this->perPage),
        ]);
    }

    // Reset all filters
    public function resetFilter()
    {
        $this->reset(['search', 'order', 'status', 'perPage', 'toDate', 'fromDate']);
    }

    // Delete deposit record
    public function delete(string $id)
    {
        Gate::authorize('delete deposits');
        try {
            $deposit = Deposit::findOrFail($id);
            if (!empty($deposit->proof)) {
                $proofPath = public_path($deposit->proof);
                if (file_exists($proofPath)) {
                    unlink($proofPath);
                }
            }
            $deposit->delete();
            Cache::forget('total_deposited');
            Cache::forget('chart_pendepsoit');
            $this->alert(type: 'success', message: 'Deposit Deleted Successfully.');
        } catch (\Throwable) {
            $this->alert(type: 'warning', message: 'Something went wrong! Deposit does not exist.');
        }
    }

    // Process deposits without external integration
    public function confirmDeposit(int $id)
    {
        Gate::authorize('process deposits');

        try {
            $deposit = Deposit::findOrFail($id);
            $settings = Cache::get('site_settings');

            // Update user's account balance and bonus
            $user = $deposit->user;
            $user->account_bal += $deposit->amount;
            
            // Apply deposit bonus if applicable
            if ($settings->deposit_bonus !== null && $settings->deposit_bonus > 0) {
                $bonusAmount = ($deposit->amount * $settings->deposit_bonus) / 100;
                $user->bonus += $bonusAmount;

                // Record the bonus transaction
                Transaction::create([
                    'user_id' => $user->id,
                    'narration' => "Deposit Bonus for {$settings->currency} {$deposit->amount} deposited",
                    'amount' => $bonusAmount,
                    'type' => 'Credit',
                ]);
            }

            $user->save();

            // Credit referral commission if applicable
            if ($settings->referral_proffit_from === 'Deposit') {
                $referralService = new ReferralCommisionService($user, $deposit->amount);
                $referralService->run();
            }

            // Update deposit status
            $deposit->status = 'Processed';
            $deposit->save();

            // Clear relevant caches
            Cache::forget('total_deposited');
            Cache::forget('chart_pendepsoit');

            // Send email notification to user with deposit details (do not let mail failures break alerts)
            if ($settings->send_deposit_email) {
                $in_app_message = "Your deposit of {$settings->currency} {$deposit->amount} has been processed.";
                try {
                    $deposit->user->notify(new DepositConfirmedNotification($deposit, $in_app_message));
                } catch (\Throwable $e) {
                    // Log silently; keep UX responsive
                    \Log::error('Admin confirm deposit notify failed', [
                        'deposit_id' => $deposit->id,
                        'user_id' => $deposit->user_id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            $this->alert(type: 'success', message: 'Deposit confirmed successfully.');
        } catch (\Throwable $th) {
            $this->alert(type: 'warning', message: $th->getMessage());
        }
    }
}
