<?php

namespace App\Livewire\Admin\Withdrawal;

use App\Models\Settings;
use App\Models\Withdrawal;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Gate;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.admin')]
class ManageWithdrawal extends Component
{
    use WithPagination;
    use LivewireAlert;
    
    #[Url(except: '')]
    public $search = '';
    public $order = 'desc';
    public $status = 'All';
    public $perPage = 10;
    public $toDate = '';
    public $fromDate = '';
    public $id = '';
    public $viewWithdrawal = false;
    
    // Edit withdrawal handling
    public $edit_id = null;
    
    protected $settings;
    
    public function mount()
    {
        try {
            $this->settings = Cache::remember('site_settings', now()->addHour(), function () {
                return Settings::first();
            });
        } catch (\Exception $e) {
            // Handle case where settings table might not exist or other DB issues
            $this->settings = null;
        }
    }
    
    public function editWithdrawal($id)
    {
        Gate::authorize('process withdrawals');
        $this->edit_id = $id;
        $this->js("$('#editWithdrawalModal').modal('show')");
    }

    #[On('refreshWithdrawals')]
    public function refreshWithdrawals()
    {
        $this->resetPage();
    }

    #[On('close-edit-withdrawal-modal')]
    public function closeEditModal()
    {
        $this->js("$('#editWithdrawalModal').modal('hide')");
        $this->edit_id = null;
    }
    
    public function render()
    {
        return view('livewire.admin.withdrawal.manage-withdrawal', [
            'withdrawals' => Withdrawal::search($this->search)
                ->status($this->status)
                ->orderBy('id', $this->order)
                ->with('user:id,name,username')
                ->date($this->fromDate, $this->toDate)
                ->paginate($this->perPage),
            'settings' => $this->settings,
        ]);
    }
    
    // Reset all filter
    public function resetFilter()
    {
        $this->reset(['search', 'order', 'status', 'perPage', 'toDate', 'fromDate']);
    }
    
    public function delete(string $id)
    {
        Gate::authorize('delete withdrawals');
        try {
            $withdrawal = Withdrawal::with('user')->findOrFail($id);
            $settings = Cache::get('site_settings');
            
            if ($settings->deduction_option === 'userRequest') {
                // Restore user's account balance if it was deducted on request
                $withdrawal->user->account_bal += $withdrawal->to_deduct;
                $withdrawal->user->save();
            }
            
            $withdrawal->delete();
            
            // Clear relevant caches
            Cache::forget('total_withdrawn');
            Cache::forget('chart_pendwithdraw');
            
            $this->alert(type: 'success', message: 'Withdrawal deleted successfully!');
        } catch (\Throwable) {
            $this->alert(type: 'warning', message: 'Something went wrong! Withdrawal record does not exist.');
        }
    }
    
    public function view(int $id): void
    {
        Gate::authorize('view withdrawals');
        $this->id = $id;
        $this->viewWithdrawal = true;
    }
    
    public function cancel()
    {
        $this->reset('id', 'viewWithdrawal');
    }
}