<?php

namespace App\Services;

use App\Exceptions\MembershipException;
use Illuminate\Support\Facades\DB;
use App\Models\Category;
use App\Models\Course;
use App\Models\Lesson;
use App\Models\UserCourse;
use App\Models\User;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;

class MembershipService
{
    public function categories(): Collection
    {
        if (Cache::has('categories')) {
            return collect(Cache::get('categories'));
        }

        $categories = Category::all();

        Cache::put('categories', $categories, now()->addHour());

        return collect($categories);
    }

    public function courses(int $page = 1): array
    {
    $query = Course::with('category', 'lessons')
        ->withCount('lessons')
        ->withCount('users')
        ->paginate(10, ['*'], 'page', $page);

    return [
        'data' => $query->items(),
        'next_page_url' => $query->nextPageUrl(),
        'prev_page_url' => $query->previousPageUrl(),
    ];
    }

    public function courseLessons(int $courseId, int $page): Collection
    {
        $query = Lesson::where('course_id', $courseId)
            ->paginate(10, ['*'], 'page', $page);

        return collect($query);
    }

    public function lessonWithoutCourse(): Collection
    {
        $lessons = Lesson::whereNull('course_id')->get();

        return collect($lessons);
    }

    public function course(int $courseId): Course
    {
        return Course::with('category', 'lessons')->findOrFail($courseId);
    }

    public function myCourses(int $userId): Collection
    {
        $userCourses = UserCourse::where('user_id', $userId)->with('course')->get();

        return collect($userCourses);
    }

    public function saveCategory(string $categoryName): string
    {
        $category = Category::create(['name' => $categoryName]);

        Cache::forget('categories');

        return "Category '{$category->name}' added successfully.";
    }

    public function deleteCategory(int $id): string
    {
        $category = Category::findOrFail($id);
        $category->delete();

        Cache::forget('categories');

        return "Category deleted successfully.";
    }

    public function addCourse(array $courseData): string
    {
        $course = Course::create($courseData);

        return "Course '{$course->title}' added successfully.";
    }

    public function updateCourse(array $courseData, int $id): string
    {
        $course = Course::findOrFail($id);
        $course->update($courseData);

        return "Course '{$course->title}' updated successfully.";
    }

    public function deleteCourse(int $id): string
    {
        
    DB::connection()->flushQueryLog();

    Course::where('id', $id)->delete();

    return "Course deleted successfully.";
    }


    public function addLesson(array $lessonData): string
    {
        $lesson = Lesson::create($lessonData);

        return "Lesson '{$lesson->title}' added successfully.";
    }

    public function updateLesson(array $lessonData, int $id): string
    {
        $lesson = Lesson::findOrFail($id);
        $lesson->update($lessonData);

        return "Lesson '{$lesson->title}' updated successfully.";
    }

    public function deleteLesson(int $id): string
    {
        $lesson = Lesson::findOrFail($id);
        $lesson->delete();

        return "Lesson deleted successfully.";
    }

    public function buyCourse(int $userId, int $courseId): string
    {
        $user = User::findOrFail($userId);
        $course = Course::findOrFail($courseId);

        if ($user->account_bal < $course->price) {
            throw new MembershipException('Insufficient balance to buy this course.');
        }

        $user->account_bal -= $course->price;
        $user->save();

        UserCourse::create([
            'user_id' => $userId,
            'course_id' => $courseId
        ]);

        return "Course '{$course->title}' purchased successfully.";
    }

    public function lessonInfo(int $lessonId): Lesson
    {
        return Lesson::findOrFail($lessonId);
    }
}
