    // Smooth scroll for anchor links
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function(e) {
            if (this.getAttribute('href') === '#') return;
            e.preventDefault();
            document.querySelector(this.getAttribute('href'))?.scrollIntoView({
                behavior: 'smooth'
            });
        });
    });

    // Navigation background on scroll
    window.addEventListener('scroll', function() {
        const nav = document.querySelector('nav');
        if (window.scrollY > 50) {
            nav.classList.add('shadow-lg');
        } else {
            nav.classList.remove('shadow-lg');
        }
    });

    // Toggle mobile menu
    const navToggle = document.getElementById('nav-toggle');
    const mobileMenu = document.getElementById('mobile-menu');
    navToggle.addEventListener('click', () => {
        mobileMenu.classList.toggle('hidden');
    });
    
      document.addEventListener('DOMContentLoaded', () => {
            const marketTableBody = document.getElementById('markets-table-body');
            const searchInput = document.getElementById('search');
            const filterButtons = document.querySelectorAll('.market-filter');
            const paginationContainer = document.getElementById('pagination');

            // All markets from fetch
            let allMarkets = [];
            let displayedMarkets = [];

            // Pagination control
            let currentPage = 1;
            const pageSize = 20;  // Adjust to show more/less per page

            // Fetch cryptocurrency data from CoinGecko API
            async function fetchMarketData() {
        try {
            // First get available pairs from Huobi
            const huobiResponse = await fetch('https://api.huobi.pro/v1/common/symbols');
            if (!huobiResponse.ok) {
                throw new Error('Failed to fetch Huobi symbols');
            }
            const huobiData = await huobiResponse.json();

            // Filter Huobi pairs for USDT pairs that are online
            const validHuobiPairs = huobiData.data
                .filter(symbol => symbol['quote-currency'] === 'usdt' && symbol.state === 'online')
                .map(symbol => ({
                    symbol: symbol['base-currency'].toUpperCase(),
                    pair: `${symbol['base-currency'].toUpperCase()}/USDT`
                }));

            // Then get CoinGecko data for images and additional info
            const coingeckoResponse = await fetch(
                'https://api.coingecko.com/api/v3/coins/markets?vs_currency=usd&order=market_cap_desc&per_page=250&page=1&sparkline=false'
            );
            const coingeckoData = await coingeckoResponse.json();

            // Create a map of symbol to CoinGecko data
            const coingeckoMap = new Map(
                coingeckoData
                    .filter(coin => 
                        !['wsteth', 'weth', 'steth'].includes(coin.symbol.toLowerCase()) &&
                        !coin.symbol.toLowerCase().startsWith('w'))
                    .map(coin => [coin.symbol.toUpperCase(), coin])
            );

            // Combine Huobi and CoinGecko data
            allMarkets = validHuobiPairs
                .filter(pair => coingeckoMap.has(pair.symbol))
                .map(pair => {
                    const coinData = coingeckoMap.get(pair.symbol);
                    return {
                        id: coinData.id,
                        pair: pair.pair,
                        name: coinData.name,
                        price: coinData.current_price.toLocaleString('en-US', { 
                            style: 'currency', 
                            currency: 'USD' 
                        }),
                        change24h: (coinData.price_change_percentage_24h || 0).toFixed(2) + '%',
                        high24h: coinData.high_24h.toLocaleString('en-US', { 
                            style: 'currency', 
                            currency: 'USD' 
                        }),
                        low24h: coinData.low_24h.toLocaleString('en-US', { 
                            style: 'currency', 
                            currency: 'USD' 
                        }),
                        volume24h: coinData.total_volume.toLocaleString('en-US', {
                            notation: 'compact',
                            compactDisplay: 'short'
                        }),
                        icon: coinData.image
                    };
                });

            // Sort by market cap (already sorted from CoinGecko)
            displayedMarkets = [...allMarkets];
            currentPage = 1;
            renderMarkets();
        } catch (error) {
            console.error('Error fetching market data:', error);
            // Optionally show an error message to the user
        }
    }

            // Renders the current page's markets
           function renderMarkets() {
    // Filter the displayedMarkets array by the current page
    const startIndex = (currentPage - 1) * pageSize;
    const endIndex = startIndex + pageSize;
    const pageData = displayedMarkets.slice(startIndex, endIndex);
    
    // Render table rows
    marketTableBody.innerHTML = pageData.map(market => {
        // Remove the slash and concatenate the pair for the URL
        const pairForUrl = market.pair.replace('/', '');
        
        return `
            <tr class="hover:bg-[#2A243B] transition-colors">
                <td class="px-4 py-3 flex items-center gap-2">
                    <div class="w-8 h-8 bg-[#2563EB]/10 rounded-lg flex items-center justify-center">
                        <img src="${market.icon}" alt="${market.name}" class="w-6 h-6 object-cover">
                    </div>
                    <div>
                        <div class="font-medium">${market.pair}</div>
                        <div class="text-xs text-gray-500">${market.name}</div>
                    </div>
                </td>
                <td class="px-4 py-3">${market.price}</td>
                <td class="px-4 py-3 ${market.change24h.startsWith('-') ? 'text-red-500' : 'text-green-500'}">
                    ${market.change24h}
                </td>
                <td class="px-4 py-3">${market.high24h}</td>
                <td class="px-4 py-3">${market.low24h}</td>
                <td class="px-4 py-3">${market.volume24h}</td>
                <td class="px-4 py-3">
                    <a href="/user/trading/trade/${pairForUrl}" class="text-[#2563EB] hover:underline text-sm">Trade</a>
                </td>
            </tr>
        `;
    }).join('');
    renderPagination();
}

            // Render pagination controls
            function renderPagination() {
                paginationContainer.innerHTML = '';

                // Number of total pages
                const totalPages = Math.ceil(displayedMarkets.length / pageSize);

                // Previous button
                const prevBtn = document.createElement('button');
                prevBtn.textContent = 'Prev';
                prevBtn.className = 'px-3 py-2 rounded border border-gray-800/50 hover:bg-[#2A243B] text-gray-300';
                prevBtn.disabled = (currentPage === 1);
                prevBtn.addEventListener('click', () => {
                    if (currentPage > 1) {
                        currentPage--;
                        renderMarkets();
                    }
                });
                paginationContainer.appendChild(prevBtn);

                // Page links (1,2,3,...)
                for (let i = 1; i <= totalPages; i++) {
                    const pageBtn = document.createElement('button');
                    pageBtn.textContent = i;
                    pageBtn.className = (i === currentPage)
                        ? 'px-3 py-2 rounded bg-[#2563EB]/10 text-[#2563EB] font-medium'
                        : 'px-3 py-2 rounded border border-gray-800/50 hover:bg-[#2A243B] text-gray-300';

                    pageBtn.addEventListener('click', () => {
                        currentPage = i;
                        renderMarkets();
                    });
                    paginationContainer.appendChild(pageBtn);
                }

                // Next button
                const nextBtn = document.createElement('button');
                nextBtn.textContent = 'Next';
                nextBtn.className = 'px-3 py-2 rounded border border-gray-800/50 hover:bg-[#2A243B] text-gray-300';
                nextBtn.disabled = (currentPage === totalPages);
                nextBtn.addEventListener('click', () => {
                    if (currentPage < totalPages) {
                        currentPage++;
                        renderMarkets();
                    }
                });
                paginationContainer.appendChild(nextBtn);
            }

            // Search functionality
            searchInput.addEventListener('input', () => {
                const searchTerm = searchInput.value.toLowerCase();
                displayedMarkets = allMarkets.filter(market => 
                    market.name.toLowerCase().includes(searchTerm) || 
                    market.pair.toLowerCase().includes(searchTerm)
                );
                currentPage = 1;
                renderMarkets();
            });

            // Filter functionality
            filterButtons.forEach(button => {
                button.addEventListener('click', () => {
                    const filter = button.getAttribute('data-filter');
                    
                    // Remove active state from all buttons
                    filterButtons.forEach(btn => {
                        btn.classList.remove('bg-[#2563EB]/10', 'text-[#2563EB]');
                        btn.classList.add('bg-transparent', 'border', 'border-gray-800/50', 'text-gray-400');
                    });

                    // Add active state to clicked button
                    button.classList.remove('bg-transparent', 'border', 'border-gray-800/50', 'text-gray-400');
                    button.classList.add('bg-[#2563EB]/10', 'text-[#2563EB]');

                    // Filter markets
                    if (filter === 'All') {
                        displayedMarkets = [...allMarkets];
                    } else {
                        displayedMarkets = allMarkets.filter(market =>
                            market.pair.includes(filter)
                        );
                    }
                    currentPage = 1;
                    renderMarkets();
                });
            });

            // Initial data fetch
            fetchMarketData();

            // Refresh data every 60 seconds
            setInterval(fetchMarketData, 60000);
        });
